import { jsx } from "react/jsx-runtime";
import { track, useEditor } from "@tldraw/editor";
import * as React from "react";
import { useAssetUrls } from "../../context/asset-urls.mjs";
import { DEFAULT_TRANSLATION } from "./defaultTranslation.mjs";
import { fetchTranslation } from "./translations.mjs";
const TranslationsContext = React.createContext(null);
function useCurrentTranslation() {
  const translations = React.useContext(TranslationsContext);
  if (!translations) {
    throw new Error("useCurrentTranslation must be used inside of <TldrawUiContextProvider />");
  }
  return translations;
}
const TranslationProvider = track(function TranslationProvider2({
  overrides,
  children
}) {
  const editor = useEditor();
  const locale = editor.user.getLocale();
  const getAssetUrl = useAssetUrls();
  const [currentTranslation, setCurrentTranslation] = React.useState(() => {
    if (overrides && overrides["en"]) {
      return {
        locale: "en",
        label: "English",
        dir: "ltr",
        messages: { ...DEFAULT_TRANSLATION, ...overrides["en"] }
      };
    }
    return {
      locale: "en",
      label: "English",
      dir: "ltr",
      messages: DEFAULT_TRANSLATION
    };
  });
  React.useEffect(() => {
    let isCancelled = false;
    async function loadTranslation() {
      const translation = await fetchTranslation(locale, getAssetUrl);
      if (translation && !isCancelled) {
        if (overrides && overrides[locale]) {
          setCurrentTranslation({
            ...translation,
            messages: { ...translation.messages, ...overrides[locale] }
          });
        } else {
          setCurrentTranslation(translation);
        }
      }
    }
    loadTranslation();
    return () => {
      isCancelled = true;
    };
  }, [getAssetUrl, locale, overrides]);
  return /* @__PURE__ */ jsx(TranslationsContext.Provider, { value: currentTranslation, children });
});
function useTranslation() {
  const translation = useCurrentTranslation();
  return React.useCallback(
    function msg(id) {
      return translation.messages[id] ?? id;
    },
    [translation]
  );
}
function untranslated(string) {
  return string;
}
export {
  TranslationProvider,
  untranslated,
  useCurrentTranslation,
  useTranslation
};
//# sourceMappingURL=useTranslation.mjs.map
